
;****************************************************************************************
;*																						*
;*	Digitales Messgert fr Drehzal und Schnittgeschwindigkeit							*
;*	----------------------------------------------------------							*
;*																						*
;*	Eine Taktscheibe die an der Motorachse befestigt ist, wird mit einer				*
;*	Gabellichtschranke abgetastet. Der Kollektor des Fototransistor ist mit dem 		*
;*	INT1 Pin des Tiny2313 verbunden. Die Taktscheibe hat 15 Bohrungen. Da sowohl		*
;*	positive als auch negative Flanken am INT0 ausgewertet werden, erhlt man 30 		*
;*	Impulse pro Umdrehung. Diese werden fr 2 Sekunden gezhlt, und man erhlt die		*
;*	Umdrehungen pro Minute (UPM). Aus UPM und dem Werkzeugdurchmesser, knnen dann die	*
;*	andere Einheiten wie z.B.Schnittgeschwindigkeit m/min berechnet werden.				*
;*	Formeln: 																			*
;*	Umdrehungen/Sek = UPM / 60															*
;*	m/min = d * pi * UPM / 1000															*
;*	m/sek = m/min / 60																	*
;*																						*
;*	Messmethode																			*
;*	-----------																			*
;*	Da eine Messung 2 Sekunden dauert, htte das aktualisieren des Displays alle 2 		*
;*	Sekunden groe Sprnge zur Folge und liee sich nur schwer ablesen. Deshalb 		*
;*	speichere ich im SRam alle 1/4 Sekunde den neusten Messwert ab und addiere die		*
;*	letzten 8 Messwerte. Eine Messung dauert so zwar immer noch 2 Sekunden allerdings	*
;*	ist eine Tendez nach oben oder unten sofort zu erkennen, wenn man die Drehzahl		*
;*	ndert.																				*
;*																						*
;*	Autor: Jrgen Woetzel www.avr-projekte.de											*
;*	Version vom 29.09.2013																*
;****************************************************************************************
#include "tn2313def.inc"
;************************************************************************************************
;Config fr bedingte Assemblierung
;************************************************************************************************
;------------------------------------------------------------------------------------------------
;Auf Unterschiedliche Art, den Gesamtmesswert bilden
;------------------------------------------------------------------------------------------------
;.set	interpol =	0	;Addieren. (Messung nicht interpolieren). 
;.set	interpol =	1	;Strkeres gewichten der neusten Messungen.
						;Die Maximale Drehzahl darf bei interpol=1 nur noch 32.767 betragen.
.set	interpol =	2	;1 Sekunde messen, Zwischenwerte interpolieren
;.set	interpol =  3	;Hochrechnen der letzten Sekunde. 1 Sekunde messen, dann *2
;.set	interpol =  4	;Hochrechnen der letzten 500mS. 1/2 Sekunde messen, dann *4, runden !
;------------------------------------------------------------------------------------------------
;Die letzte Stelle runden. Beruhigt die Anzeige auf Kosten der Auflsung
;------------------------------------------------------------------------------------------------
.set	genau	=	0	;Wenn 1, nicht runden. Wenn 0, die Letzte Ziffer = 0 oder 5 
.set	strong	=	0	;Starkes runden. Wenn strong=1 UND genau=0 zeigt die letzte Stelle stndig Null
;------------------------------------------------------------------------------------------------
;Startwerte
;------------------------------------------------------------------------------------------------
.set	startdm	=	30	;Werkzeugdurchmesser(*10) beim einschalten 30=3,0 120=12
.set	everon	=	0	;Werkzeugdurchmesser in allen Modi anzeigen (0=aus)
.set	everchange = 0	;Werkzeugdurchmesser in allen Modi einstellbar (0=aus)
;*************************************************************************************************
;---------------------------------------------------------
;Register
;---------------------------------------------------------
.def	null	=	r0		;Konstante 0
.def	stat	=	r1		;Statusregister
.def	cnt_low	=	r2		;Zhler int1 (Flanken der Lichtschranke)
.def	cnt_high=	r3
.def	Pointer	=	r4		;Zeiger auf Messwert
.def	ms		=	r5		;Zhler Timerinterrupt
.def	taste1	=	r6		;Merkt sich den Tastenzustand im Timerinterrupt (entprellen)
.def	taste2	=	r7
.def	taste3	=	r8
.def	pi_mal_d_1	=r9		;Zwischenergebniss (3,14*d) 24Bit
.def	pi_mal_d_2	=r10
.def	pi_mal_d_3	=r11
.def	tick		=r12	;Zhler Timerinterrupt
.def	repeat		=r13	;Repeatzhler Up -Downtaste
.def	sechzig		=r14	;Konstante

.def	temp1	=	r16		;Arbeitsregister
.def	temp2	=	r17
.def	temp3	=	r18
.def	temp4	=	r19
.def	temp5	=	r20
.def	itemp	=	r21		;Arbeitsregister Timerinterrupt
.def	durchmesser	=	r22	;Werkzeugdurchmesser *10 (Festkomma, eine Stelle)
.def	flags		=	r23	;Programmflags
.equ	error		=	7	;berlauf
.equ	enter		=	6	;Entprellte Tasten
.equ	down		=	5
.equ	up			=	4
.equ	runden		=	3	;Wird gesetzt wenn bei Division /10 der Rest >=5 ist
.equ	mode1		=	1	;Entscheiden den Anzeigemodus (UPM, m/min usw.)
.equ	mode0		=	0
;--------------------------------------------------------------------------
;SRamadressen
;--------------------------------------------------------------------------
.dseg
.org	$60
Segbuffer:		.byte	8	;Inhalt der 7 Anzeigen und der 4 LEDs.
CntBuffer:		.byte	16	;8 Words (low/high). Jedes Wort enthlt 1/8 (250mS) der Gesamtmessung (2Sek.) 
;--------------------------------------------------------------------------
.cseg
.org	0
		rjmp	init
.org	INT1addr
		rjmp	Ext_int
.org	OC1Aaddr			;Timer0 Overflow Interrupt
		rjmp	Timer_int
;**************************************************************************
;							Externer Interrupt (int1)
;**************************************************************************
Ext_int:
		;-------------------------------------------------------------------------------
		;Eine Codierscheibe mit 15 Bohrungen wird von einer Gabellichschranke abgetastet.
		;Da der INT1 Interrupt bei steigender und fallender Flanke aufgerufen wird, erhlt
		;man 30 Aufrufe pro Umdrehung. Zhlt man fr genau 2 Sekunden die Aufrufe, erhlt
		;man die Umdrehungen pro Minute (UPM)
		;-------------------------------------------------------------------------------
		in		stat,SREG			;Status sichern
		inc		cnt_low				;Flanken der Lichtschranke zhlen
		brne	ext1
		inc		cnt_high
;		brne	ext1
;		sbr		flags,1<<error
ext1:	out		SREG,stat			;Status wieder zurck
		reti		
;**************************************************************************
;							Timer Interrupt (max. 8S im Simulator)
;**************************************************************************
;Im Simulator dauert die ISR maximal 8 S. Somit ist die max. abtastbare Frequenz 125 kHz,
;ohne Impulse zu verschlucken. Bei 60000 U/min betrgt die Frequenz 60000/60 = 1000Hz.
;1000 * 30 (Die Bohrungen der Lochscheibe * 2) = 30 kHz. Keine Gefahr also.
;-----------------------------------------------------------------------------------------
Timer_Int:
		in		stat,SREG			;Status sichern
		push	temp1
		;-------------------------------------------------------------------------------
		;Alle 250mS die gezhlten Impulse der Lichtschranke im SRam (8 Words) speichern.
		;Diese werden dann in der Main addiert und man erhlt die Anzahl der Impulse 
		;der letzten 2 Sekunden
		;-------------------------------------------------------------------------------
		dec		ms					;250mS	herunterzhlen
		brne	mux					;Ist noch was da? Wenn ja,Sprung
		ldi		itemp,250
		mov		ms,itemp			;die nchsten 250mS vorbereiten
		mov		itemp,pointer		;Pointer +2 (Wg. Word) 
		subi	itemp,-2
		andi	itemp,$0f			;Max. 15 zulassen, dann wieder bei 0 anfangen
		mov		pointer,itemp
		ldi		yl,low(CntBuffer)	;Zeiger auf nchstes Wort im SRam
		ldi		yh,high(CntBuffer)
		add		yl,pointer
		adc		yh,null
		st		Y+,cnt_low			;Anzahl der Impulse wegspeichern
		st		Y,cnt_high
		clr		cnt_low				;Zhler Resetten
		clr		cnt_high
		dec		tick
		;---------------------------------------------------
		;7-Segment muxen
		;---------------------------------------------------
mux:	clr		itemp
		out		PortB,itemp					;Segmente aus
		in		itemp,PinD					;Dekoder hochzhlen
		mov		temp1,itemp
		subi	itemp,-3					;+3, Reihenfolge der Digits = 0,3,6,1,4,7,2,5,0 .....
		andi	itemp,1<<pd0|1<<pd1|1<<pd2	;Den 3-Bit Wert des Dekoders Ausmaskieren
		in		temp1,portd
		andi	temp1,0b11111000			;Alle anderen Bits von PortD unberhrt lassen
		or		temp1,itemp				
		out		PortD,temp1					;Dekoder auf nchstes Digit schalten
		ldi		yl,low(Segbuffer)			;Zeiger auf das Segment in temp1
		ldi		yh,high(Segbuffer)
		add		yl,itemp
		adc		yh,null						;Knnte man weglassen
		ld		temp1,Y						;Das passendes Byte aus dem Buffer holen
		out		PortB,temp1					;Segmente schalten
		;-----------------------------------
		;Tasten an PortD (6,5,4) entprellen
		;-----------------------------------
debounce:in		itemp,pind
		lsl		itemp					;S1 ins 7. Bit
		;-----------------------------------
		;Mode Taster PinD6
		;-----------------------------------
		lsl		itemp					;S1 ins Carry
		rol		taste1					;S1 (Carry) in Taste1 (Bit0) schieben
		brcc	deb1					;Es muss eine 1 (nicht gedrckt) und danach 8 mal eine 0
		brne	deb1					;(gedrckt) folgen, erst dann ist der Tastendruck gltig.          
		sbr		flags,1<<enter			;Flag setzen, wird in der Main, nach Auswertung wieder gelscht.
		;----------------------
		;Taste <UP> mit Repeat
		;----------------------
deb1:	lsl		itemp					;Up-Taste ins Carry
		rol		taste2					;und in Taste2 schieben
		brne	deb2					;Taste ist 0 wenn die letzten 8 Durchlufe gedrckt war			
		dec		repeat					;Repeatzhler-1
		brne	rp0						;Wenn Repeatzeit noch nicht abgelaufen, auf neuen Tastendruck testen
		mov		repeat,sechzig			;Repeatwert (kurz) neu laden
		sec								;Tastenflag durch Repeat setzen
rp0:	brcc	deb2					;Das Carry setzt das Tastenflag durch Repeat oder ersten Tastendruck				
		sbr		flags,1<<up	
		;-----------------------
		;Taste <DOWN> mit Repeat
		;-----------------------
deb2:	lsl		itemp					;Down-Taste ins Carry
		rol		taste3					;und in Taste schieben
		brne	deb3					;Taste ist 0 wenn die letzten 8 Durchlufe gedrckt war			
		dec		repeat					;Repeatzhler-1
		brne	rp1						;Wenn Repeatzeit noch nicht abgelaufen, auf neuen Tastendruck testen
		mov		repeat,sechzig			;Repeatwert (kurz) neu laden
		sec								;Tastenflag durch Repeat setzen
rp1:	brcc	deb3					;Das Carry setzt das Tastenflag durch Repeat oder ersten Tastendruck				
		sbr		flags,1<<down
		;----------------------------------------------------------------------
		;Wenn keine Taste gedrckt ist Repeatwert (1. Tastendruck = lang) laden
		;----------------------------------------------------------------------
deb3:	tst		taste2
		breq	deb4
		tst		taste3
		breq	deb4
		clr		repeat					;256 mS (= ca.1/4 Sekunde)
		;-----------------------
		;Interruptende einleiten
		;-----------------------
deb4:	pop		temp1					;Register fr die Main wieder Herstellen
		out		SREG,stat				;Status wieder zurck
		reti		
;**************************************************************************
;						 Init
;**************************************************************************
init:	ldi 	temp1, RAMEND				;Stack
		out 	SPL, temp1
		;---------------------------------------------------
		;Ports
		;---------------------------------------------------
		ldi		temp1,$ff					;PotrtB, Segmente (Kathoden) auf Ausgang
		out		ddrb,temp1
		ldi		temp1,1<<pd0|1<<pd1|1<<pd2	;Gem.Anoden ber Decoder, auf Ausgang
		out		ddrd,temp1
		ldi		temp1,1<<pd3|1<<pd4|1<<pd5|1<<pd6
		out		portd,temp1					;Pullups fr die 3 Taster und Lichtschranke Einschalten
		;---------------------------------------------------
		;Startwerte Programm
		;---------------------------------------------------
		clr		null						;Konstante
		ldi		temp1,60
		mov		sechzig,temp1				;Tastenrepeatwert kurz
		ldi		durchmesser,startdm			;Beim Start ist der Werkzeugdurchmesser = 3,0 mm
		rcall	print_dm					;Werkzeugdurchmesser anzeigen
		clr		pointer						;Zeigt auf einen der 8 Messwerte
		clr		flags						;Programmflags
		clr		temp1
		rcall	modeled						;UPM-LED Einschalten
		;---------------------------------------------------
		;Timerinterupt anwerfen
		;---------------------------------------------------
		ldi		temp1,high(8000-1)		;Vergleichswert =8000 ergibt
		out		OCR1AH,temp1			;bei 8MHz genau 1000 Hz
		ldi		temp1,low(8000-1)
		out		OCR1AL,temp1
		ldi		temp1,(1<<WGM12)|(1<<CS10);Prescaler=1 Timer Compare ein
		out		tccr1b,temp1
		ldi		temp1,1<<ocie1a			;Interrupt bei Timer Compare ein
		out		TIMSK,temp1				
		;---------------------------------------------------
		;INT1 Interupt anwerfen
		;---------------------------------------------------
		ldi		temp1,1<<ISC10			;Beide Flanke erzeugen einen Interrupt
	 	out		MCUCR,temp1
		ldi		temp1,1<<INT1
		out		GIMSK,temp1				;Interrupt bei Flanke an INT1 ein
		sei
		rcall	welcome					;Mind. 2 Sekunden vertrdeln

;**************************************************************************
;							 Mainloop
;**************************************************************************
main:	rcall	tsterror				;Test auf berlauf (65536 UPM)
		rcall	print_dm				;D6,D7 (Segmente fr Werkzeugdurchmesser) ein/ausschalten
		rcall	taste					;Mode Taster abfragen
		.if		everchange ==1
		rcall	tasteup					;Werkzeugdurchmesser mit Tasten hoch/runterzhlen
		rcall	tastedown
		.endif
		sbrc	flags,mode1				;Mode1 Bit entscheidet ob Schnittgeschwindigkeit 
		rjmp	m3_m4					;oder Umdrehung/Zeit angezeigt wird
		sbrc	flags,mode0				;Mode0 Bit entscheidet, ob UPM oder UPS angezeigt wird
		rjmp	printups				;Umdrehungen pro Sekunde anzeigen
		rjmp	printupm				;Umdrehungen pro Minute
m3_m4:	.if		everchange ==0
		rcall	tasteup					;Werkzeugdurchmesser mit Tasten hoch/runterzhlen
		rcall	tastedown
		.endif
		sbrc	flags,mode0
		rjmp	printms					;m/min anzeigen
		rjmp	printmm					;m/sek anzeigen
;**************************************************************************
;Unterprogramme
;**************************************************************************
tsterror:
		sbrs	flags,error				;Wenn das Errorflag gesetzt ist, berspringen
		ret								;Ansonsten: Alles klar, Rcksprung
		cbr		flags,1<<error			;Errorflag lschen
		ldi		temp1,_e				;Das Wort Error aufs Display
		sts		segbuffer,temp1
		ldi		temp1,_r
		sts		segbuffer+1,temp1
		sts		segbuffer+2,temp1
		sts		segbuffer+4,temp1
		ldi		temp1,_0o
		sts		segbuffer+3,temp1
w05:	ldi		temp1,2					;1/2 Sekunde warten
		rcall	wt1
		ret
		;---------------------------------------------------
		;Mode Taste abfragen
		;---------------------------------------------------
taste:	sbrs	flags, enter				;Wenn Tastenflag gesetzt, berspringen
		ret									;Ansonsten rcksprung
		cbr		flags,1<<enter|1<<up|1<<down;alle Tastenflags lschen
		mov		temp1,flags					;Modusbits im Flagregister (Bit0 und1) erhhen
		inc		temp1
		andi	temp1,0b11					;Nur 4 Modi erlauben
		andi	flags,0b11111100			;alten Modus lschen
		or		flags,temp1					;neuen Modus speichern
		;---------------------------------------------------------------
		;Passende Modus LED schalten. (2Bit Binr zu Dezimal wandeln +1)
		;---------------------------------------------------------------
modeled:clr		temp2						;Dezimalwert =0						;
		sec		
llp:	rol		temp2						;Dezimalwert +1
		dec		temp1						;Binrwert -1
		brpl	llp							;Solange wiederholen bis Binrwert= -1
		sts		segbuffer+7,temp2			;Dezimalwert +1 ins LED-Register (wird mit den Anzeigen gemuxt)
		ret
		;---------------------------------------------------
		;Tasten fr Werkzeugdurchmesser abfragen & auswerten
		;---------------------------------------------------
TasteUp:sbrs	flags, up			;Wenn Tastenflag gesetzt, berspringen
		ret							;Ansonsten rcksprung
		cbr		flags,1<<up			;Tastenflag lschen
		cpi		durchmesser,100		;Durchmesser >10 mit Dezimalpunkt (1/10 mm anzeigen)
		brcc	ab10				;Durchmesser + 1mm
		inc		durchmesser			;Durchmesser + 0,1mm
		rcall	print_dm
		ret
ab10:	cpi		durchmesser,250		;Maximum (25,0mm) erreicht ?
		brcc	maxdm				;Wenn ja, Sprung
		subi	durchmesser,-10		;ansonsten Durchmesser + 1,0 mm
		rcall	print_dm			;Aufs Display damit
maxdm:	ret

TasteDown:
		sbrs	flags, down			;Wenn Tastenflag gesetzt, berspringen
		ret							;Ansonsten rcksprung
		cbr		flags,1<<down		;Tastenflag lschen
		cpi		durchmesser,101
		brcc	ab010
		cpi		durchmesser,1
		breq	maxdm
		dec		durchmesser
		rcall	print_dm
		ret
ab010:	subi	durchmesser,10
		rcall	print_dm
		ret
		;--------------------------------------------------------
		;Werkzeugdurchmesser auf den 2 Anzeigen (D6, D7) ausgeben
		;--------------------------------------------------------
print_dm:
		.if		everon ==0
		sbrs	flags,mode1			;Bei UPM und UPS die Anzeigen ausschalten
		rjmp	blank
		.endif
		cpi		durchmesser,100		;Bei Durchmesser grsser 10, kein Dezimalpunkt
		brcc	pdm10
		;--------------------------------------------------------
		;Werkzeugdurchmesser von 0,1..9,9 ausgeben
		;--------------------------------------------------------
		mov		temp3,durchmesser	;Division vorbereiten
		clr		temp4
		rcall	div10				;Durch 10 Teilen, der Rest der Division wird schon
		sts		segbuffer+6,temp1	;als 7-Segmencode geliefert (temp1)
		rcall	div10				;Nochmal durch 10
		sbr		temp1,1<<7			;Dezimalpunkt dazuodern
		sts		segbuffer+5,temp1	;und in D6 schreiben	
		ret
		;--------------------------------------------------------
		;Werkzeugdurchmesser von 10..25 (ohne Dezimalpunkt)
		;--------------------------------------------------------
pdm10:	mov		temp3,durchmesser	;Division vorbereiten
		clr		temp4
		rcall	div10				;Erste Division verwerfen (Kommastelle)
		rcall	div10
		sts		segbuffer+6,temp1
		rcall	div10
		sts		segbuffer+5,temp1
		ret
		;--------------------------------------------------------
		;Werkzeugdurchmesser aus
		;--------------------------------------------------------
blank:	clr		temp1				;0=Leerzeichen
		sts		segbuffer+6,temp1
		sts		segbuffer+5,temp1
		ret
		;-----------------------------------------
		;Delay
		;-----------------------------------------
wt1:	mov		tick,temp1			;tick wirt im Timerinterrupt alle 250mS heruntergezhlt
wait:	tst		tick
		brne	wait
		ret
;**************************************************************************
;Berechnen und anzeigen der verschiedenen Modi
;**************************************************************************
		;--------------------------------------------------------
		;UPM aufs Display
		;--------------------------------------------------------
printupm:rcall	upm					;Die letzten 8 Messwerte (Impulse pro 250mS) addieren
		rcall	div10				;Durch 10 teilen (Rest der div. wird von div10 in 7Seg-Code gewandelt)
		.if	genau ==0				;Wenn gewnscht, letzte Stelle runden
		ldi		temp1,_0
		.if	strong ==0				;Wenn gewnscht, eine Null in die letzte Stelle
		sbrc	flags,runden		;runden wird von div10 gesetzt wenn Rest >=5
		ldi		temp1,_5
		.endif
		.endif
		sts		segbuffer+4,temp1	;In die letzte Stelle des Displays schreiben	
		rcall	test0				;Divident = 0 -> Leerzeichen. Ansonsten zu div10 springen
		sts		segbuffer+3,temp1	;In der Vorletzten Stelle eintragen
		rcall	test0				;Weiter wie gehabt
		sts		segbuffer+2,temp1
		rcall	test0
		sts		segbuffer+1,temp1
		rcall	test0
		sts		segbuffer,temp1
		rjmp	main
		;--------------------------------------------------------
		;U/sekunde 
		;UPM/6, + Dezimalpunkt eine Stelle nach links (enspricht /60)
		;--------------------------------------------------------
printups:rcall	upm					;Die letzten 8 Messwerte (Impulse pro 250mS) addieren
		rcall	div6				;Da 10tel angezeigt werden, nur durch 6 anstatt 60 teilen
		rcall	div10				;1/10tel berechnen
		sts		segbuffer+4,temp1	;1/10tel in die letzte Stelle
		rcall	div10				;Einer
		sbr		temp1,1<<7			;Dezimalpunkt der vorletzten Anzeige einschalten
		sts		segbuffer+3,temp1
		rcall	test0
		sts		segbuffer+2,temp1
		rcall	test0
		sts		segbuffer+1,temp1
		rcall	test0
		sts		segbuffer,temp1
		rjmp	main
		;-------------------------------------------------------------------------------------------
		;Meter pro Minute
		;-------------------------------------------------------------------------------------------
		;Die Formel n*d*pi/1000 wurde erweitert, um in ASM mit Festkomma rechnen zu knnen (2 Nachkommastellen)
		;Die neue Formel lautet n*(d*10)*(pi*655,36)/$10000
		;n=Umdrehungen/Minute (UPM)
		;d=Werkzeugdurchmesser .Dieser liegt als Festkommazahl mit einer Stelle nach dem Komma (*10) 
		;in r22 (durchmesser) vor.
		;pi = die Keiszahl 3,14.....Diese wird mit 655,36 ($1000/100dez) multipliziert. Dadurch kann die
		;anschliesende Division /1000 entfallen, es werden einfach die beiden untersten Bytes verworfen.
		;
		;Die endgltige Formel mit Programmvariablen:
		;temp1..5 =durchmesser*2059*upm mit anschliesendem verwerfen der Bytes temp1 und temp2.
		;-------------------------------------------------------------------------------------------
printmm:rcall	mul2059				;durchmesser(*10) mal 2059 (pi*655,36)
		rcall	mul_upm				;Ergebniss mal upm
		rcall	div010				;Ergebniss in einzelne Ziffern (7-Seg) wandeln und ausgeben
		.if	genau ==0				;Wenn gewnscht, letzte Stelle runden
		ldi		temp1,_0
		.if	strong ==0				;Wenn gewnscht, eine Null in die letzte Stelle
		sbrc	flags,runden
		ldi		temp1,_5			;Wenn gewnscht 0 oder 5 in die letzte Stelle
		.endif
		.endif
		sts		segbuffer+4,temp1
		rcall	div010
		sts		segbuffer+3,temp1
		rcall	div010
		sbr		temp1,1<<7			;2 Stellen nach dem Komma den DP einschalten
		sts		segbuffer+2,temp1                                                                     
		rcall	test0
		sts		segbuffer+1,temp1
		rcall	test0
		sts		segbuffer,temp1
		rjmp	main
		;-------------------------------------------------------------------------------------------
		;Meter pro Sekunde
		;-------------------------------------------------------------------------------------------
		;Formel wie oben, nur mit anschliesender Division durch 6. Durch die Division durch 6 anstatt
		;durch 60, erhlt man eine Stelle hinter dem Komma mehr als bei m/min. Hier also 1000stel
		;-------------------------------------------------------------------------------------------
printms:rcall	mul2059				;durchmesser(*10) mal 2059 (pi*655,36)
		rcall	mul_upm				;Ergebniss * upm
		rcall	div6				;Ergebniss / 6
		rcall	div10				;In einzelne Ziffern (7-Seg) wandeln und ausgeben
		sts		segbuffer+4,temp1
		rcall	div10
		sts		segbuffer+3,temp1
		rcall	div10
		sts		segbuffer+2,temp1                                                                     
		rcall	div10
		sbr		temp1,1<<7			;3 Stellen nach dem Komma den DP dazuodern
		sts		segbuffer+1,temp1
		rcall	test0
		sts		segbuffer,temp1
		rjmp	main
;**************************************************************************
;Aus den 8 Messwerten auf Unterschiedliche Art, das Gesamtergebniss bilden
;**************************************************************************	
		;------------------------------------------------------------------------------------------
		;Wenn man am Frsmotor die Drehzahl ndert, reagiert bei normaler Messung die Anzeige etwas
		;trge, da eine genaue Messung 2 volle Sekunden dauert. Um dem entgegezuwirken kann man die
		;neueren Messungen strker gewichten als ltere Messungen. Das Extrem whre, die letzte Messung 
		;mit 8 zu Multiplizieren. Das htte allerdings zur Folge, das auch die Auflsung auf 8 
		;steigen wrde (man htte also nur ganzzahlige Werte der 8er Reihe auf dem Display).
		;Die Lsung unten ist also ein Kompromiss zwischen Auflsung und der Reaktion auf 
		;Drehzahlnderungen der Anzeige.
		;------------------------------------------------------------------------------------------
		;Die 8 Messungen werden wie folgt gewichtet:
		;------------------------------------------------------------------------------------------
		;Neueste.....................................................................lteste Messung
		;	1			2			3			4			5			6			7			8
		;------------------------------------------------------------------------------------------
		;31,25%		  18,75%	  12,5%		  12,5%		   6,25%	   6,25%	   6,25%	   6,25%
		;   5			3			2			2			1			1			1			1
		;------------------------------------------------------------------------------------------
		;In der unteren Zeile der Tab ist die Anzahl der Messungen eingetragen die zum Gesamtwert
		;addiert werden.
		;Zuerst werden alle 8 Messungen zum Gesamtwert addiert. Dann wird der neuste Wert 4mal 
		;dazuaddiert, der nchste 2 mal, der 3. und 4. Wert 1mal.
		;Zum Schluss wird der Gesamtwert durch 2 geteilt.
		;------------------------------------------------------------------------------------------
upm:	.if	interpol ==1
		rcall	_upm				;Alle Messwerte der letzten 2 Sekunden addieren
		mov		temp5,pointer		;Zeigt auf den letzten Messwert
		subi	temp5,6				;Gehe 3 Words zurck
		andi	temp5,$0f			;Unterlauf interresiert nicht (Ringbuffer)
		ldi		xl,low(CntBuffer)	;Zeiger auf Messwert vor 1 Sekunde setzen
		ldi		xh,high(CntBuffer)
		add		xl,temp5			;Index dazu
		adc		xh,null				;Carry dazu
		rcall	adm					;Messwert (vor 1 Sekunde) addieren
		rcall	adm					;Messwert (vor 0,75 Sekunde) addieren
		rcall	adm					;Messwert (vor 0,5 Sekunden) 2* addieren
		rcall	am5
		rcall	adm					;Messwert (vor 0,25 Sekunden) 4* addieren
		rcall	am5
		rcall	am5
		rcall	am5
		brcc	ok1
		sbr		flags,1<<error		;Bei berlauf Fehlerflag setzen
		ret
ok1:	lsr		temp4				;Gesamtwert durch 2 teilen
		ror		temp3
		ret
		.endif
		;-------------------------------------------------------------------------------------------------
		;Die letzten 1000 mS Auswerten und Zwischenwerte interpolieren.
		;Andere Taktik als oben.
		;Es werden die letzten 4 Messungen addiert und 4 mal der Mittelwert zweier benachbarten Messungen.
		;Dadurch dauert eine Gesamtmessung nur noch 1 Sekunde anstatt 2
		;-------------------------------------------------------------------------------------------------
		.if	interpol ==2
		mov		temp5,pointer
		subi	temp5,8				;Gehe 4 Messungen zurck
		rcall	addm+1				;Messung vor 1,25 Sek. nach zl:zh
		clr		temp3				;Gesamtmessung = 0
		clr		temp4
		rcall	addm				;4* Gesamtwert und Mittelwert zur Gesamtmessung addieren			
		rcall	addm
		rcall	addm
		rcall	addm
		ret
		;------------------------------------------------
		;Pointer +2, Messwert -> temp1..2
		;------------------------------------------------
addm:	subi	temp5,-2
		andi	temp5,$0f	
		ldi		xl,low(CntBuffer)	
		ldi		xh,high(CntBuffer)
		add		xl,temp5			;Index dazu
		adc		xh,null				;Carry dazu
		ld		temp1,x+			;Messwert nach temp1..2
		ld		temp2,x
		;------------------------------------------------
		;Mittelwert und ganzer Messwert zum Gesamtergebniss addieren
		;------------------------------------------------
		add		zl,temp1			;Mittelwert aus Z und Temp1..2 bilden (z= alter Wert)
		adc		zh,temp2			;interpolierter Wert = alter Wert + neuer Wert / 2
		lsr		zh
		ror		zl
		add		temp3,zl			;Gesamtwert + interpolierter Wert
		adc		temp4,zh
		mov		zl,temp1			;neuer alter Wert ;-)			
		mov		zh,temp2
		add		temp3,zl			;Gesamtwert + neuer Wert
		adc		temp4,zh
		ret
		.endif
		;------------------------------------------------
		;Nur die letzten 1000 mS Auswerten und Hochrechnen
		;------------------------------------------------
		.if	interpol ==3
		mov		temp5,pointer		;Zeigt auf den letzten Messwert
		subi	temp5,6				;3 Words zurck
		andi	temp5,$0f			;Unterlauf interresiert nicht (Ringbuffer)
		ldi		xl,low(CntBuffer)	;Zeiger auf Messwert vor 1 Sekunde setzen
		ldi		xh,high(CntBuffer)
		add		xl,temp5			;Index dazu
		adc		xh,null				;Carry dazu
		clr		temp3
		clr		temp4
		rcall	adm
		rcall	adm
		rcall	adm
		rcall	adm
		lsl		temp3
		rol		temp4
		ret
		.endif
		;------------------------------------------------
		;Nur die letzten 500 mS Auswerten und Hochrechnen
		;------------------------------------------------
		.if	interpol ==4
		mov		temp5,pointer		;Zeigt auf den letzten Messwert
		subi	temp5,2				;1 Word zurck
		andi	temp5,$0f			;Unterlauf interresiert nicht (Ringbuffer)
		ldi		xl,low(CntBuffer)	;Zeiger auf Messwert vor 1/2 Sekunde setzen
		ldi		xh,high(CntBuffer)
		add		xl,temp5			;Index dazu
		adc		xh,null				;Carry dazu
		clr		temp3
		clr		temp4
		rcall	adm
		rcall	adm
		lsl		temp3
		rol		temp4
		lsl		temp3
		rol		temp4
		ret
		.endif
		;----------------------------------------------
		;Messwert addieren und Zeiger im CntBuffer erhhen
		;----------------------------------------------
		.if		interpol >0
adm:	cli							;Bitte nicht stren
		ld		temp1,x+			;Messwert nach temp1 und 2
		ld		temp2,x+
		sei
		add		temp3,temp1			;Zum gesamtmesswert addieren
		adc		temp4,temp2
		subi	temp5,-2			;Index +2
		andi	temp5,$0f			;berlauf weg unden
		ldi		xl,low(CntBuffer)	;Zeiger auf Bufferanfang
		ldi		xh,high(CntBuffer)
		add		xl,temp5			;Index und Carry dazu
		adc		xh,null
		ret
		;----------------------------------------------
		;Messwert nochmal addieren
		;----------------------------------------------
am5:	add		temp3,temp1
		adc		temp4,temp2
		ret
		.endif
		;------------------------------------------------------------------------------------------
		;Die letzten 8 Messwerte addieren
		;------------------------------------------------------------------------------------------
		;Der CntBuffer (8 Words) wird im Timerinterrupt alle 250 mS mit den gezhlten
		;Impulsen der Taktscheibe beschieben. Es wird immer der lteste Wert mit dem 
		;neusten berschrieben (Ringbuffer).	
		;------------------------------------------------------------------------------------------
		.if		interpol <=1
_upm:	ldi		xl,low(CntBuffer)	;X zeigt auf den CntBuffer
		ldi		xh,high(CntBuffer)
		clr		temp3				;Ergebniss =0
		clr		temp4
		ldi		temp5,8				;8 Additionen
addupm:	cli
		ld		temp1,x+			;1. Word nach temp1..2
		ld		temp2,x+
		sei
		add		temp3,temp1			;addieren
		adc		temp4,temp2
		dec		temp5				;8 Additionen durch ?
		brne	addupm				;Wenn nein, Sprung
		brcc	ok
		sbr		flags,1<<error		;Bei berlauf Fehlerflag setzen
ok:		ret
		.endif
;**************************************************************************
;Mathe
;**************************************************************************	
		;------------------------------------------------------------------------------------------
		;Nullen vor der Dezimalzahl ausblenden
		;------------------------------------------------------------------------------------------
		;Ist der Dividend 0, wird anstelle der Ziffer null ein Leerzeichen ausgegeben. Ansonsten
		;wird durch 10 dividiert.
		;------------------------------------------------------------------------------------------
test0:	cp		temp3,null			;Ist noch was da?
		cpc		temp4,null
		breq	f0					;Wenn nein, Sprung
		rjmp	div10				;Dividieren
	f0:	clr		temp1				;Leerzeichen in temp1
		ret
		;----------------------------------------------
		;16Bit Division durch 6 (8Bit Konstante)
		;Dividend und Ergebniss in temp3 und temp4
        ;Rest in temp1
		;-----------------------------------------
div6:	clr		temp1			;Rest
		ldi		temp2,16		;16* schieben
div61:	lsl		temp3			;Solange Bits vom Divident in Rest schieben bis >= 6 ...			
		rol		temp4
		rol		temp1			;Rest
		cpi		temp1,6
		brcs	div62			;Wenn der Rest <6 ist, 0-Bit (LSL oben) im Ergebniss stehen lassen
		inc		temp3			;Ansonsten im Ergebniss Bit0=1 ...
		subi	temp1,6			;.. und 6 vom Rest abziehen
div62:	dec		temp2			;das ganze 16 mal
		brne	div61
		ret
		;----------------------------------------------
		;24Bit Division durch 10
		;Dividend und Ergebniss in temp3..5
        ;Rest in temp1
		;-----------------------------------------
div010:	clr		temp1			;Rest
		ldi		temp2,24		;24* schieben
div0101:lsl		temp3			;Solange Bits vom Divident in Rest schieben bis >= 10 ...			
		rol		temp4
		rol		temp5
		rol		temp1			;Rest
		cpi		temp1,10
		brcs	div0102			;Wenn der Rest <10 ist, 0-Bit (LSL oben) im Ergebniss stehen lassen
		inc		temp3			;Ansonsten im Ergebniss Bit0=1 ...
		subi	temp1,10		;.. und 10 vom Rest abziehen
div0102:dec		temp2			;das ganze 24 mal
		brne	div0101
		rjmp	seg7			;Rest nach 7Seg wandeln

		;----------------------------------------------
		;16Bit Division durch 10
		;Dividend und Ergebniss in temp3 und temp4
        ;Rest in temp1
		;-----------------------------------------
div10:	clr		temp1			;Rest
		ldi		temp2,16		;16* schieben
div101:	lsl		temp3			;Solange Bits vom Divident in Rest schieben bis >= 10 ...			
		rol		temp4
		rol		temp1			;Rest
		cpi		temp1,10
		brcs	div102			;Wenn der Rest <10 ist, 0-Bit (LSL oben) im Ergebniss stehen lassen
		inc		temp3			;Ansonsten im Ergebniss Bit0=1 ...
		subi	temp1,10		;.. und 10 vom Rest abziehen
div102:	dec		temp2			;das ganze 16 mal
		brne	div101
;		ret
		;-----------------------------------------
		;Ende Division
		;Aus dem Rest, den 7-Segmentcode ermitteln
		;-----------------------------------------
seg7:	cbr		flags,1<<runden
		cpi		temp1,5
		brcs	seg71
		sbr		flags,1<<runden
seg71:	ldi		zl,low(segtab*2)	;Z auf 7-Segmenttab.
		ldi		zh,high(segtab*2)
		add		zl,temp1			;+ Rest
		adc		zh,null				;Carry dazu
		lpm		temp1,Z
		ret
		;-----------------------------------------
		;Multiplikation Durchmesser(*10) * 2059 (pi*655,36)
		;8Bit * 16Bit - Konstante
		;Ergebniss in temp1..3 (24Bit)
		;-----------------------------------------
mul2059:mov		temp4,durchmesser
		ldi		temp5,8				;8* schieben
		clr		temp1
		clr		temp2
		clr		temp3
mul1:	lsl		temp1				;Ergebniss schieben
		rol		temp2
		rol		temp3
		lsl		temp4				;Faktor schieben, Carry entscheidet ob addiert wird
		brcc	mul2				;Sprung wenn geschobenes Bit=0 war
		subi	temp1,low(-2059)	;Ansonsten 2059 addieren
		sbci	temp2,high(-2059)
		sbci	temp3,-1			;Carry dazu
mul2:	dec		temp5				;Alle 8 Bit durch ?
		brne	mul1
		mov		pi_mal_d_1,temp1	;Ergebniss wegspeichern (low Register)
		mov		pi_mal_d_2,temp2
		mov		pi_mal_d_3,temp3
		ret
		;-----------------------------------------
		;Multiplikation D*Pi (schon multipliziert) * UPM
		;24Bit * 16Bit
		;Ergebniss in temp1..5 (40Bit)
		;-----------------------------------------
mul_upm:rcall	upm
		mov		xl,temp3
		mov		xh,temp4			;UPM nach X
		ldi		zl,16				;16* schieben
		clr		temp1				;Ergebniss lschen
		clr		temp2
		clr		temp3
		clr		temp4
		clr		temp5
mul01:	lsl		temp1				;Ergebniss schieben
		rol		temp2
		rol		temp3
		rol		temp4
		rol		temp5
		lsl		xl					;Faktor schieben, Carry entscheidet ob addiert wird
		rol		xh
		brcc	mul02
		add		temp1,pi_mal_d_1	;24Bit addieren
		adc		temp2,pi_mal_d_2
		adc		temp3,pi_mal_d_3
		adc		temp4,null
		adc		temp5,null
mul02:	dec		zl					;das ganze 16 mal
		brne	mul01
		ret
		;------------------------------------------------------------------------------
		;Laufschrift Hallo (Poweron)
		;Da beim Einschalten noch unbestimmte Werte im CntBuffer stehen, die ersten 2
		;Sekunden mit einer Laufschrift vertrdeln
		;-------------------------------------------------------------------------------
wtext:	.db	0,0,0,0,0,_h,_a,_l,_l,_o,0,0,0,0,0,$ff	;7-Seg Codes fur Hallo
		;-------------------------------------------------------------------------------
welcome:ldi		zl,low(wtext<<1)	;Z zeigt auf den Text
		ldi		zh,high(wtext<<1)
sc2:	lpm		temp2,z+			;Die Zeichen nacheinander aufs Display (D1..D5)
		sts		segbuffer,temp2
		lpm		temp2,z+
		sts		segbuffer+1,temp2
		lpm		temp2,z+
		sts		segbuffer+2,temp2
		lpm		temp2,z+
		sts		segbuffer+3,temp2
		lpm		temp2,z+
		tst		temp2				
		brmi	exscr				;$ff Markiert das Ende des Strings
		sts		segbuffer+4,temp2
		ldi		temp1,1				;Zeit=250mS
		mov		tick,temp1
wait1:	tst		tick				;Tick wird im Timerinterrupt alle 250 mS heruntergezhlt
		brne	wait1
		sbiw	z,4					;Um zu scrollen 4 von Z subtrahieren (Z steht auf dem 5. Zeichen)
		rjmp	sc2					;Noch ne Runde
exscr:	ret
;**************************************************************************
;Ende Code
;**************************************************************************	
segtab:
.db 	 0x3F,0x06
.db 	 0x5B,0x4F
.db 	 0x66,0x6D
.db 	 0x7D,0x07
.db 	 0x7F,0x6F
.db 	 0x77,0x7C
.db 	 0x58,0x5E
.db 	 0x79,0x71


.equ	_0     =  	 $3F	; 0
.equ	_1     =  	 $06	; 1
.equ	_2     =  	 $5B	; 2
.equ	_3     =  	 $4F	; 3
.equ	_4     =  	 $66	; 4
.equ	_5     =  	 $6D	; 5
.equ	_6     =  	 $7D	; 6
.equ	_7     =  	 $07	; 7
.equ	_8     =  	 $7F	; 8
.equ	_9     =  	 $6F	; 9
.equ	_a     =  	 $77	; A
.equ	_b     =  	 $7C	; B
.equ	_c     =  	 $58	; C
.equ	_d     =  	 $5E	; D
.equ	_e     =  	 $79	; E
.equ	_f     =  	 $71	; F
.equ	_0g    =  	 $6F	; g
.equ	_g     =  	 $3D	; G
.equ	_0h    =  	 $74	; h
.equ	_h     =  	 $76	; H
.equ	_0i    =  	 $04	; i
.equ	_i     =  	 $06	; I
.equ	_j     =  	 $1E	; J
.equ	_l     =  	 $38	; L
.equ 	_0n    =  	 $54	; n
.equ	_n     =  	 $37	; N
.equ	_0o    =  	 $5C	; o
.equ	_o     =  	 $3F	; O
.equ	_p     =  	 $73	; P
.equ 	_q     =  	 $67	; q
.equ	_r     =  	 $50	; r
.equ	_s     =  	 $6D	; S
.equ	_t     =  	 $78	; t
.equ	_0u    =  	 $1C	; u
.equ	_u     =  	 $3E	; U
.equ	_y     =  	 $72	; Y
