;****************************************************************************************
;*	DDM (Digitales Druckmessgert)                                                      *
;*	------------------------------                                                      *
;*  Software: Einen Drucksensor mit dem internen ADC des ATmega auslesen und wahlweise  *
;*            in Bar, PSI oder kPa (Kilopascal) anzeigen. Umschalten der 3 Modi ber    *
;*            einen Taster. Modusanzeige ber 2 Leds. Nullpunktabgleich ber einen 2.   *
;*            Taster mit anschliessendem speichern im Eeprom.                           *
;*                                                                                      *
;*	Hardware: Platine mit AT-Mega48 http://www.avr-projekte.de                      	*
;*            Drucksensor MPX5700, Ausgang an PC0 (ADC0)                                *
;*            3 Siebensegmentanzeigen, gem.Anode. Kathoden an PortD; Anoden an PB0..2   *
;*            2 Taster an PC1 und 2. 2 LEDs an PC3 und 4                                *
;*                                                                                      *
;*	Version vom 12.08.2012                                                              *
;*                                                                                      *
;*	Autor: J.Woetzel / juergen@avr-projekte.de                                          *
;****************************************************************************************
#include "m48def.inc"
;Referenzspannung
;----------------
.equ	vref	= 1<<REFS0			;5V intern
;---------------------------------------------------------------------------------------------------
.def	null	=	r0		;Konstante
.def    taste   =   r1
.def    taste2  =   r2
.def    offset  =   r3
.def	tick	=	r4
.def	s		=	r5		;S-Register
.def    ylow    =   r6
.def    yhigh   =   r7
.def    tled    =   r8
.def    ltemp   =   r9
.def	temp1	=	r16		;Trash       
.def	temp2	=	r17       
.def	temp3	=	r18       
.def	temp4	=	r19
.def	temp5	=	r25
.def	itemp	=	r20		;ISR-Variable
.def    itemp2  =   r10
.def	seg0	=	r21		;7Seg. Digits, Hunderter
.def	seg1	=	r22		;Zehner
.def	seg2	=	r23		;Einer
;----------------------------------------------------------------------------------------------------
.def    flags   =   r24
;------------------------
.equ    modeb       =   0
.equ    modekpa     =   1
.equ    key         =   2
.equ    key_offset  =   3
.equ    err         =   4
.equ    dp1         =   5
.equ    dp2         =   6
.equ    blink       =   7
;----------------------------------------------------------------------------------------------------
.equ    Q1          =   pb0 ;3 PNP-Transistoren zum einschalten der Segmentanzeigen
.equ    Q2          =   pb1
.equ    Q3          =   pb2
.equ    LED1        =   pc3 ;LEDs zum Anzeigen von Bar/PSI
.equ    LED2        =   pc4
.equ    S1          =   pc2 ;Taster zum Nullabgleich
.equ    S2          =   pc1 ;Taster zum Wechsel
;-----------------------------------------------------------------------------------------------------
.org 0
	rjmp	res
;***************************************************************************************************************
;ISR, ca. alle 2mS
;***************************************************************************************************************
.org OVF0addr
ISR:in		s,sreg
	inc		tick		
	;-----------------------------------------------------
	;Display muxen
	;Bei 3 Anzeigen *2mS sind das 6mS fr einen Durchlauf.
	;Die Muxfreq. ist somit 1/6mS = 166,66 Hz
	;-----------------------------------------------------
	ser		itemp			
	out		Portd,itemp				;Segmente aus

	sbic	PortB,Q3				;War zuletzt die 3. Anzeige eingeschaltet, 1. Anzeige einschalten
	rjmp	_mux1
    sbi     PortB,Q3                ;3. Display aus
    out     PortD,seg0              ;Segmente in PortD laden
    cbi     PortB,Q1                ;1. Display ein
    sbrc    flags,dp1               ;Evtl. Dezimalpunkt schalten
    cbi     PortD,7
	rjmp	bl1
_mux1:
	sbic	PortB,Q1				;Wenn D1 an war, D2 einschalten
	rjmp	_mux2
    sbi     PortB,Q1
    out     PortD,seg1
    cbi     PortB,Q2
    sbrc    flags,dp2               ;Evtl. Dezimalpunkt schalten
    cbi     PortD,7
	rjmp	bl1
_mux2:
    sbi     PortB,Q2				;kann nur noch Stelle 3 sein
	out		PortD,seg2
	cbi     PortB,Q3
	;-----------------------------------------------------------------------------------------
    ;Tasten an PortC entprellen (nur jede 3. ISR, alle 6 mS)
    ;Bei gedrcktem Taster wird (nach dem Kontaktprellen) nach 8 ISRs (48mS) taste=0 sein 
    ;und das Carry=1. Nur dann, wird das Tastenflag gesetzt. 
	;-----------------------------------------------------------------------------------------
t1: lsl     taste                   ;Eine 0 in taste schieben
    sbic    PinC,S1                 ;Bei gedrckter Taste, berspringen
    inc     taste                   ;Wenn Taster nicht gedrckt, die 0 in eine 1 wandeln
    brne    t2                      ;Wenn der Taster die letzten 8 ISRs nicht gehalten wurde, Sprung
    brcc    t2                      ;Wurde der Taster vo 9 ISRs noch nicht gehalten, ist der Tastendruck gltig                  
    sbr     flags,1<<key_offset     ;Tastenflag setzten
t2: lsl     taste2
    sbic    PinC,S2
    inc     taste2
    brne    bl1
    brcc    bl1
    sbr     flags,1<<key  
	;-------------------------------------------------------
    ;Wenn Blinkflag=1 die LEDs abwechselnd Ein & Ausschalten
	;-------------------------------------------------------
bl1:sbrs    flags,blink             ;Wenn blink=0, nicht blinken
    rjmp    bl2
    tst     tick                    ;Alle 256 ISRs (ca.0,5 Sek.) die LEDs togglen
    brne    exit
    in      itemp,portc
    eor     itemp,tled              ;tled = 1<<LED1|1<<LED2 = 0b00011000
    out     portc,itemp
    rjmp    exit
	;-------------------------------------------------------
    ;Wenn Blinkflag=0 die LEDs anhand des Modus schalten
	;-------------------------------------------------------
bl2:in      itemp2,portc
    or      itemp2,tled             ;0b00011000 Leds erstmal aus (negative Logik)
    sbrc    flags,modeb
    ldi     itemp,~(1<<led2)        ;0b11101111 Bei Bar, LED2 ein
    sbrs    flags,modeb
    ldi     itemp,~(1<<led1)        ;0b11110111 Bei PSI, LED1 ein
    sbrc    flags,modekpa
    ldi     itemp,~(1<<led1|1<<led2);0b11100111 Bei Kilopascal, beide LEDs ein
    and     itemp2,itemp
    out     portc,itemp2
	;-----------------------------------------------------
    ;ISR Ende
	;-----------------------------------------------------
exit:
	out		sreg,s					;Statusbits zurcklesen
	reti							;in der Main weitermachen
;***************************************************************************************************************
;									--- ENDE ISR ---
;***************************************************************************************************************														
;										Init
;***************************************************************************************************************
	;-------------------------------------
	;Ports init
	;-------------------------------------
res:ldi		temp1,$ff
	out		ddrd,temp1				;Segmente (Kathoden) Ausgang
	ldi		temp1,1<<Q1|1<<Q2|1<<Q3
	out		ddrb,temp1				;gem. Anoden Ausgang
	ldi		temp1,1<<LED1|1<<LED2
    out     ddrc,temp1              ;LEDs Ausgang
    ldi     temp1,1<<S1|1<<S2
    out     PortC,temp1             ;Pullups der Taster einschalten
	;-------------------------------------
	;Timerinterrupt init
	;-------------------------------------
	ldi		temp1,1<<cs00|1<<cs01	;TC0/64, ISR ca. alle 2mS (8MHz/256/64)
	out		TCCR0B,temp1	
	ldi		temp1,1<<TOIE0			;Timer0 overflow interrupt
	sts		TIMSK0,temp1
	;-------------------------------------
	;ADC initialisieren
	;-------------------------------------
	ldi		temp1,vref				; Kanal und vref wird oben definiert
	sts		ADMUX,temp1
	ldi		temp1,1<<ADPS1|1<<ADPS2	|1<<ADEN
	sts		ADCSRA,temp1 			;Prescaler /64=125 khz
	ldi		temp1,1<<ADC0D
	sts		DIDR0,temp1				;PortC0 Digitaleingang aus
    rcall   sample1
	;-------------------------------------
    ser     seg0                    ;Segmente aus
    ser     seg1
    ser     seg2

    ldi     temp1,1<<LED1|1<<LED2
    mov     tled,temp1
	clr		null					;Konstante
    clr     flags
    rcall   ReadEE                  ;Letzten Wert vom Nullabgleich auslesen
	sei                             ;ISR los
;************************************************************************************************
;*	Hauptschleife	
;************************************************************************************************
main:
	rcall	wait			;Um die Anzeige zu beruhigen, 400 mS vertrdeln
    rcall   cal             ;Auf Nullabgleich testen

setm:
    rcall   setmode         ;Taste abragen und evtl. Modus Bar/PSI/kPa toggeln
    sbrc    flags,key       ;Falls gerade gewechselt wurde, gleich nochmal abfragen
    rjmp    setm

	rcall	sample			;Analogwert auslesen(Mittelwert aus 256 Messungen)
    rcall   tsterror        ;Auf Fehler (Druck unter null) testen   
    sbrc    flags,err
    rjmp    main            ;Bei Fehler, neuer Versuch

    rcall   mulx            ;Analogwert in Bar/PSI/kPa umrechnen           
    rcall   dez             ;Nach Dezimal wandeln und aufs Display schreiben

	rjmp	main            ;neue Runde

;************************************************************************************************
;Nullabgleich, Offset
;Es darf kein Druck am Sensor anliegen
;************************************************************************************************
cal:sbrs    flags,key_offset    ;Taster gedrckt?
    ret                         ;Wenn nein, wars das
    cbr     flags,1<<key_offset ;Tastenflag lschen
    cli                         ;LEDs in der ISR abwechselnd blinken lassen
    sbi     portc,LED2          ;Grundstellung= LED1 ein, LED2 aus
    cbi     portc,LED1
    sbr     flags,1<<blink      ;Blinkflag (ISR) setzten
    sei
    ;--------------------------------------------------------------
    ;"CAL" aufs Display, DPs aus
    ;--------------------------------------------------------------
    cbr     flags,1<<dp1|1<<dp2
    ldi     seg0,_C
    ldi     seg1,_A
    ldi     seg2,_L
    ;--------------------------------------------------------------
    ;Innerhalb 1,2 Sekunden 2*messen und das Ergebniss vergleichen
    ;--------------------------------------------------------------
ca1:
    clr     offset          ;offset=0
    rcall   sample          ;1. Messung
    tst     yh              ;Offset >255?
    brne    cal             ;Kann nicht sein, nochmal
    push    yl              ;Analogwert wegspeichern
    rcall   wait1200
    rcall   sample          ;2. Messung
    pop     offset          ;1. Messung zurck
    cp      yl,offset       ;Messung1=Messung2?
    brne    ca1             ;Da hat sich was bewegt, nochmal das ganze
    ;------------------------------------------------------------------
    ;Da das letzte Bit des A/D-Wandlers wackelt 1 vom A/D-Wert abziehen
    ;Die Anzeige ist dann bei Bar, immer noch "0.00" 
    ;Wird bei PSI wieder korrigiert (ADC-1), ansonsten zeigt PSI 00.1
    ;------------------------------------------------------------------
    dec     offset
    ;--------------------------------------------------------------
    ;Offset als Dezimalziffer auf die Anzeige
    ;--------------------------------------------------------------
    ldi     seg0,_oo        ;Kleines "o" in die erste Stelle
    mov     temp3,offset    ;Zum Dividieren durch 10 vorbereiten
    clr     temp4
    rcall   div10           ;/10
    lpm     seg2,z          ;In die letzte Stelle des Display
    rcall   div10           ;/10 
    lpm     seg1,z          ;In die mitte
    rcall   wait1200
    ;--------------------------------------------------------------
    ;Ins Eeprom schreiben und Ende
    ;--------------------------------------------------------------
    rcall   WriteEE         ;Weg damit
    ldi     seg0,_E         ;End" aufs Display
    ldi     seg1,_nn
    ldi     seg2,_d
    rcall   wait1200
    cbr     flags,1<<blink  ;Leds wieder auf den Modus schalten
    ret
    ;--------------------------------------------------------------
    ;EEprom	schreiben
    ;--------------------------------------------------------------
WriteEE:
	cli
    clr     temp2
WriteEE1:
	sbic	EECR,EEWE			;Bereit ?
	rjmp	WriteEE1
	out		EEARL,temp2			;Adresse
	out		EEDR,offset			;Daten
	cli
	sbi		EECR,EEMWE			;Master Write Enable
	sbi		EECR,EEWE			;Write Enable
	sei
	ret
    ;--------------------------------------------------------------
    ;EEprom	LESEN
    ;--------------------------------------------------------------
ReadEE:
    clr     temp2				;Adresse 0x00
nxtEE:
    sbic	EECR,EEWE			;Bereit ?
	rjmp	nxtEE
	out		EEARL,temp2			;Adresse setzen
	sbi		EECR,EERE			;ReadEnable in Controllregister
	in		offset,EEDR			;Daten Holen
	ret
;************************************************************************************************
;	Fehlertest (Unterdruck)
;************************************************************************************************
tsterror:
    cbr     flags,1<<err
    tst     yh                  ;Bei negativem A/D-Wert muss ein Nullabgleich durchgefhrt werden
    brpl    ok                  ;Bei Postitv, alles Klar
    ldi     seg0,g_             ;3 Minuszeichen (Segment g) anzeigen
    ldi     seg1,g_
    ldi     seg2,g_
    sbr     flags,1<<err        ;Fehlerflag setzen
ok: ret
;************************************************************************************************
;	Taste (S2, Bar/PSI/kPa) abfragen und evtl. Mode ndern
;************************************************************************************************
setmode:
    sbrs    flags,key                       ;Taste gedrckt?
    ret                                     ;Wenn nein, Fertig
    cbr     flags,1<<key|1<<dp1|1<<dp2      ;Tastenflag, Dezimalpunkte lschen da Text aufs Display soll

    inc     flags                           ;Die beiden mode-Flags als 2-Bit Zhler nutzen
    mov     temp1,flags                     ;Zum vergleichen vorbereiten
    andi    temp1,1<<modeb|1<<modekpa       ;Die restlichen Flags, wegmaskieren
    cpi     temp1,1<<modeb|1<<modekpa       ;Nur Mode 0..2 Zulassen
    brne    set1
    cbr     flags,1<<modeb|1<<modekpa       ;Bei 3, Zhler resetten
set1:
    sbrs    flags,modekpa                   ;Modus kurz auf Display zeigen
    rjmp    set2
    ldi     seg0,_P                         ;"PAS" fr kPa aufs Display
    ldi     seg1,_A                         ;Eigentlich falsch, aber ein 7Segment kann kein K :-(        
    ldi     seg2,_S
    rjmp    sm4
set2:
    sbrc    flags,modeb
    rjmp    set3
    ldi     seg0,_B
    ldi     seg1,_A
    ldi     seg2,_R
    rjmp    sm4
set3:
    ldi     seg0,_P
    ldi     seg1,_S
    ldi     seg2,_I

sm4:rcall   wait                            ;Moduswechsel ca. 0,8 Sek. anzeigen
    rcall   wait
    ret
;************************************************************************************************
;	AD-Wandler
;************************************************************************************************
;-------------------------------------
;256 Messungen
;-------------------------------------
sample:
	clr		ltemp			;Den 24-Bit Wert nach yh,yl,temp1
	clr		yl
	clr		yh
	clr		temp4 			;Zhler
	;-------------------------------------------------------
	;256 Messungen, dann Mittelwert bilden (temp1 verwerfen)
	;-------------------------------------------------------
sa1:rcall	sample1			;1 mal messen..
	add		ltemp,xl
	adc		yl,xh			;.. und addieren
	adc		yh,null
	dec		temp4			;Messung-1
	brne	sa1				;Letzte Messung ? Wenn nein, Sprung
sa2:sub 	yl,offset   	;Offset abziehen
	sbci	yh,0
    ret
	;------------------------------------------------------
	;A/D Wandler fr eine Messung starten
    ;10-Bit Wert in X
	;------------------------------------------------------
sample1:
	lds		temp5,ADCSRA	;ADC Los
	sbr		temp5,1<<ADSC
	sts		ADCSRA,temp5
wait_adc:
	lds		temp5,ADCSRA
    andi    temp5, 1<<ADSC 
	brne	wait_adc		;ADSC Bit pollen
	lds		xl, ADCL		;ADC-Wert wegspeichern
	lds		xh, ADCH
	ret
;************************************************************************************************
;	Delays
;************************************************************************************************
    ;--------------------------------------------------------------
    ;1,2 Sekunden Warten
    ;--------------------------------------------------------------
wait1200:
    rcall   wait            ;400mS warten
    rcall   wait
	;-------------------------------------
	;Zwischen den Messungen 400mS warten
	;-------------------------------------
wait:
    push    temp1
    ldi     temp1,55
	mov		tick,temp1		;Wird in der ISR alle 2mS hochgezhlt
w1: sbrc    flags,key       ;Bei Tastendruck, die Warteschleife vorzeitig verlassen
    rjmp    ex
	tst		tick            ;Warten bis die ISR tick hochgezhlt hat
	brne	w1
ex: pop     temp1
	ret
;------------------------------------------------------------------------------
;Abhngig von den Mode-Flags, den AD-Wert in Bar, Kilopascal oder PSI berechnen
;------------------------------------------------------------------------------
mulx:   sbrc    flags,modekpa
        rjmp    mulKPa
        sbrs    flags,modeb
        rjmp    mulBar
;-------------------------------------------------------------------------
;PSI berechnen
;-------------------------------------------------------------------------
mulPSI: sbiw    y,1             ;AD-Wert -1 (den Offset +1 bei PSI korrigieren)
        brpl    ps1             ;Aber nur, wen wir positiv bleiben (das letzte ADC-Bit wackelt)
        adiw    y,1
        /*
        --------------------------------------------------------------------------------------------------------
        Der MPX5700 hat eine Range von 4,5V. Bei vref=5V sind das 1024*4,5/5 = 921,6 (gekrzt wird zum Schluss)
        Bei 921,6 am ADC betrgt der Druck 700 kPa, das entspricht 101,526416411 PSI(*). Da eine Stelle hinter dem
        Komma angezeigt werden soll sind das pro ADC Schritt 101,526416411 * 10 / 921,6 
        = 1,1016321225151909722222222222222. Die Auflsung (921) reicht also nicht ganz aus um alle 
        1/10 PSI Werte Darzustellen.
        Um den gemessenen ADC-Wert mit 1,1016 zu multiplizieren rechne ich Nachkommastelle * ADC + ADC.
        Das entspricht bei einem AD-Wert von z.B. 100:
        100 * 0,1016 + 100 = 110,16
        Um in ASM mit 0,1016 multiplizieren zu knnen wird zuerst mit $10000 multipliziert und beim 32 Bit-Ergebniss
        die unteren 2 Bytes verworfen.
        0,1016321225151909722222222222222 * $10000 = 6660,5. Das Ergebniss ist schlecht zu runden, deshalb
        rechne ich 0,1016321225151909722222222222222 * 2 *$10000 = 13321,125 ~ 13321 und teile das Ergebniss 
        durch $10000 und danach durch 2.

        (*)Quelle:http://www.umrechnung.org/masseinheiten-druck-umrechnen/einheiten-umrechner-druck-pressure.htm
        --------------------------------------------------------------------------------------------------------
        */
ps1:    cbr     flags,1<<dp1    ;Dezimalpunkt 1 aus    
        movw    yhigh:ylow,Y    ;A-D Wert wegspeichern       
        ldi     zl,low(13321)   ;0,1016 *2 *$10000 = 13321
        ldi     zh,high(13321)   
        rcall   mul16           ;Ergebniss (32Bit) Temp1..4, Temp1 und 2 werden Verworfen (/$10000)
        lsr     temp4           ;Da 0,1016 ungnstig zu multiplizieren ist, wurde mit (0,1016*2) 
        ror     temp3           ;multipliziert. Deshalb jetzt /2
        add     temp3,ylow      ;AD-Wert addieren
        adc     temp4,yhigh
        ;-------------------------------------
        ;Auf 100 PSI testen, und ggf. verzweigen
        ;-------------------------------------
        ldi     temp1,high(1000)
        cpi     temp3,low(1000) ;Ergebniss mit 1000 (PSI*10) vergleichen
        cpc     temp4,temp1
        brcs    psi1            ;Wenn kleiner, Sprung
        ;-------------------------------------
        ;Ab 100 PSI, keine 1/10 mehr anzeigen
        ;-------------------------------------
        cbr     flags,1<<dp2    ;Dezimalpunkt fr 1/10 ausschalten
        rcall   div10           ;PSI*10 durch 10 teilen um den ganzzahligen Wert anzuzeigen
        ret
        ;-------------------------------------
        ;Unter 100 PSI, 1/10 anzeigen
        ;-------------------------------------
psi1:   sbr     flags,1<<dp2    ;Dezimalpunkt fr 1/10 ein
        ret
;-------------------------------------------------------------------------
;Kilopascal berechnen
;-------------------------------------------------------------------------
mulKPa: cbr     flags,1<<dp1|1<<dp2
        rjmp    mb
;-------------------------------------------------------------------------
;Bar berechnen
;-------------------------------------------------------------------------
/*
Bei 921,6 (siehe PSI) zeigt die Anzeige 7.00 Bar (Bar*100)
700 / 921,6 = 0,75954861111111111111111111111111 * $10000 = 49777,77 ~ 49778
*/

mulBar: sbr     flags,1<<dp1
        cbr     flags,1<<dp2
mb:     ldi     zl,low(49778)
        ldi     zh,high(49778)
        rcall   mul16
        ret
;-------------------------------------------------------------------------
;Multiplikation 16*16=32 Bit. Faktor1=Y, Faktor2=Z, Ergebniss temp1..temp4
;-------------------------------------------------------------------------
mul16:	ldi		temp5,16		;16*schieben
		clr		temp1			;Ergebnis (Produkt)lschen
		clr		temp2
		clr		temp3	
		clr		temp4
mul01:	lsl		temp1			;Ergebniss schieben
		rol		temp2
		rol		temp3
		rol		temp4
		lsl		yl				;Herausgeschobenes Bit aus Multiplikator (ADC)..
		rol		yh 				;..entscheidet ob addiert wird
		brcc	mul02			;Wenn Bit=0 nicht addieren
		add		temp1,zl		;Wenn Bit=1 Multiplikant (Formel) zum Ergebniss addieren
		adc		temp2,zh
		adc		temp3,null
		adc		temp4,null
mul02:	dec		temp5			;Alle 16 Bit durch ?
		brne	mul01
		ret
;**********************************************************************************************
;Dezimal ausgeben
;**********************************************************************************************
dez:	rcall	div10
		lpm		seg2,z			
		rcall	div10
		lpm		seg1,z			
		rcall	div10
		lpm		seg0,z			
        ret
		;----------------------------------------------
		;16Bit Division durch 10
		;Dividend und Ergebniss in temp3 und temp4
        ;Rest in temp1
		;-----------------------------------------
div10:	clr		temp1			;Rest
		ldi		temp2,16		;16* schieben
div101:	lsl		temp3			;Solange Bits vom Divident in Rest schieben bis >= 10 ...			
		rol		temp4
		rol		temp1			;Rest
		cpi		temp1,10
		brcs	div102			;Wenn der Rest <10 ist, 0-Bit (LSL oben) im Ergebniss stehen lassen
		inc		temp3			;Ansonsten im Ergebniss Bit0=1 ...
		subi	temp1,10		;.. und 10 vom Rest abziehen
div102:	dec		temp2			;das ganze 16 mal
		brne	div101
		;-----------------------------------------
		;Ende Division
		;Aus dem Rest, den 7-Segmentcode ermitteln
		;-----------------------------------------
seg7:	ldi		zl,low(segtab*2)	;Z auf 7-Segmenttab.
		ldi		zh,high(segtab*2)
		add		zl,temp1			;+ Rest
		adc		zh,null				;Carry dazu
		ret
;**********************************************************************************************
;Ende Code
;**********************************************************************************************
;Segment | PortD
;-----------------------------------------------------------------------------------------------
;	a		0		|-a-|	
;	b		1		f   b	Durch den gemeinsamen Anodenanschluss, ist eine Null am Port ntig
;	c		2		|-g-|	damit ein Segment leuchtet
;	d		3		e   c
;	e		4		|-d-|.(dp)
;	f		5
;	g		6
;   dp      7
;-------------------------------------------------------------------
;Die Zeichen 0..F als 7Segmentcode
segtab: 
.db   $C0, $F9, $A4, $B0, $99, $92, $82, $F8, $80, $90, $88, $83, $A7, $A1, $86, $8E
space:
.db   $ff,$ff
;-------------------------------------------
;Einzelne Segmente
;-------------------------------------------
.equ    g_ = 0b10111111
.equ    f_ = 0b11011111
.equ    e_ = 0b11101111
.equ    d_ = 0b11110111
.equ    c_ = 0b11111011
.equ    b_ = 0b11111101
.equ    a_ = 0b11111110
;-------------------------------------------
;Equates der Chars
.equ   _0 =   	 $C0	; 0
.equ   _1 =   	 $F9	; 1
.equ   _2 =   	 $A4	; 2
.equ   _3 =   	 $B0	; 3
.equ   _4 =   	 $99	; 4
.equ   _5 =   	 $92	; 5
.equ   _6 =   	 $82	; 6
.equ   _7 =   	 $F8	; 7
.equ   _8 =   	 $80	; 8
.equ   _9 =   	 $90	; 9
.equ   _a =   	 $88	; A
.equ   _b =   	 $83	; B
.equ   _cc=   	 $A7	; C
.equ   _c =   	 $C6	; C
.equ   _d =   	 $A1	; D
.equ   _e =   	 $86	; E
.equ   _f =   	 $8E	; F
.equ   _gg=   	 $90	; g
.equ   _g =   	 $C2	; G
.equ   _hh=   	 $8B	; h
.equ   _h =   	 $89	; H
.equ   _ii=   	 $FB	; i
.equ   _i =   	 $F9	; I
.equ   _j =   	 $E1	; J
.equ   _l =   	 $C7	; L
.equ   _nn=   	 $AB	; n
.equ   _n =   	 $C8	; N
.equ   _oo=   	 $A3	; o
.equ   _o =   	 $C0	; O
.equ   _p =   	 $8C	; P
.equ   _q =   	 $98	; q
.equ   _r =   	 $AF	; r
.equ   _s =   	 $92	; S
.equ   _t =   	 $87	; t
.equ   _uu=   	 $E3	; u
.equ   _u =   	 $C1	; U
.equ   _y =   	 $8D	; Y
;-------------------------------------------
.eseg
.db 33
